// server.js
const express = require('express');
const { google } = require('googleapis');
const { JWT } = require('google-auth-library');
const cors = require('cors');
require('dotenv').config();

// Load credentials from JSON file
const credentials = require('./credentials.json');

const app = express();
app.use(cors());
app.use(express.json());

class GoogleSheetsService {
  constructor(credentials) {
    this.credentials = credentials;
    this.client = null;
    this.sheets = null;
  }

  async initialize() {
    try {
      this.client = new JWT({
        email: this.credentials.client_email,
        key: this.credentials.private_key,
        scopes: ['https://www.googleapis.com/auth/spreadsheets.readonly']
      });

      this.sheets = google.sheets({ version: 'v4', auth: this.client });
      console.log('Google Sheets service initialized successfully');
    } catch (error) {
      console.error('Error initializing Google Sheets service:', error);
      throw error;
    }
  }

  parseDateTime(dateStr, timeStr) {
    try {
      console.log('Parsing datetime for:', { dateStr, timeStr });

      // Split the date components
      const [month, day, year] = dateStr.split('/').map(num => parseInt(num, 10));
      
      // Parse the time string (format: "3:00:00 PM")
      let [time, period] = timeStr.split(' ');
      let [hours, minutes, seconds] = time.split(':').map(num => parseInt(num, 10));
      
      // Convert to 24-hour format if PM
      if (period === 'PM' && hours !== 12) {
        hours += 12;
      } else if (period === 'AM' && hours === 12) {
        hours = 0;
      }

      // Validate the components
      if (isNaN(year) || isNaN(month) || isNaN(day) || 
          isNaN(hours) || isNaN(minutes) || isNaN(seconds)) {
        throw new Error('Invalid date/time components');
      }

      // Get current year in both Gregorian and Buddhist calendar
      const currentGregorianYear = new Date().getFullYear();
      const currentBuddhistYear = currentGregorianYear + 543;

      // Convert year if it's in Buddhist calendar (greater than current Buddhist year)
      let finalYear = year;
      if (year > currentGregorianYear && year <= currentBuddhistYear) {
        finalYear = year - 543;
        console.log('Converting from Buddhist year:', year, 'to Gregorian year:', finalYear);
      }

      // Create date string in ISO format to ensure proper parsing
      const dateTimeStr = `${finalYear}-${String(month).padStart(2, '0')}-${String(day).padStart(2, '0')}T${String(hours).padStart(2, '0')}:${String(minutes).padStart(2, '0')}:${String(seconds).padStart(2, '0')}`;
      const finalDate = new Date(dateTimeStr);

      console.log('Parsed result:', {
        originalDate: dateStr,
        originalTime: timeStr,
        originalYear: year,
        convertedYear: finalYear,
        convertedTime: `${hours}:${minutes}:${seconds}`,
        finalDateTime: finalDate,
        finalDateTimeISO: finalDate.toISOString()
      });

      return finalDate;
    } catch (error) {
      console.error('Error parsing date/time:', { dateStr, timeStr, error });
      return null;
    }
  }

  async getMeetings(spreadsheetId) {
    try {
      // Get data from QUERY sheet
      const response = await this.sheets.spreadsheets.values.get({
        spreadsheetId,
        range: 'QUERY!A1:J1000', // Adjust range as needed
      });

      const rows = response.data.values;
      if (!rows || rows.length === 0) {
        console.log('No data found in sheet');
        return { currentMeeting: null, upcomingMeetings: [] };
      }

      const headers = rows[0];
      console.log('Headers:', headers);

      const meetings = rows.slice(1).map(row => {
        const meeting = {
          timestamp: row[headers.indexOf('ประทับเวลา')],
          email: row[headers.indexOf('ที่อยู่อีเมล')],
          organizer: row[headers.indexOf('ชื่อ-สกุล')],
          department: row[headers.indexOf('ฝ่าย')],
          title: row[headers.indexOf('หัวข้อการประชุม')],
          attendees: row[headers.indexOf('จำนวนผู้เข้าร่วม')],
          startDate: row[headers.indexOf('วันที่เริ่มต้น')],
          startTime: row[headers.indexOf('เวลาที่เริ่ม')],
          endDate: row[headers.indexOf('วันที่สิ้นสุด')],
          endTime: row[headers.indexOf('เวลาที่สิ้นสุด')]
        };
        
        // Add parsed dates
        meeting.parsedStartDateTime = this.parseDateTime(meeting.startDate, meeting.startTime);
        meeting.parsedEndDateTime = this.parseDateTime(meeting.endDate, meeting.endTime);
        
        return meeting;
      }).filter(meeting => meeting.parsedStartDateTime && meeting.parsedEndDateTime); // Filter out any meetings with invalid dates

      // Get current time
      const now = new Date();
      console.log('\n--- Current time ---');
      console.log('Current time:', now);
      console.log('Current time (ISO):', now.toISOString());
      
      // Log all meetings data
      console.log('\n--- All Meetings Data ---');
      meetings.forEach((meeting, index) => {
        console.log(`\nMeeting ${index + 1}:`);
        console.log('Title:', meeting.title);
        console.log('Raw Start Date:', meeting.startDate);
        console.log('Raw Start Time:', meeting.startTime);
        console.log('Parsed Start DateTime:', meeting.parsedStartDateTime);
        console.log('Raw End Date:', meeting.endDate);
        console.log('Raw End Time:', meeting.endTime);
        console.log('Parsed End DateTime:', meeting.parsedEndDateTime);
      });

      // Get current meeting
      const currentMeeting = meetings.find(meeting => {
        return now >= meeting.parsedStartDateTime && now <= meeting.parsedEndDateTime;
      });

      if (currentMeeting) {
        console.log('Found current meeting:', currentMeeting.title);
      }

      // Get upcoming meetings
      console.log('\n--- Finding Upcoming Meetings ---');
      const upcomingMeetings = meetings
        .filter(meeting => {
          const isUpcoming = meeting.parsedStartDateTime > now;
          console.log(`Meeting "${meeting.title}" is upcoming:`, isUpcoming);
          console.log('Start time:', meeting.parsedStartDateTime);
          console.log('Comparison with now:', {
            meetingTime: meeting.parsedStartDateTime.getTime(),
            nowTime: now.getTime(),
            difference: meeting.parsedStartDateTime.getTime() - now.getTime()
          });
          return isUpcoming;
        })
        .sort((a, b) => a.parsedStartDateTime - b.parsedStartDateTime)
        .slice(0, 5);

      console.log('Found upcoming meetings:', upcomingMeetings.length);

      return {
        currentMeeting: currentMeeting ? {
          title: currentMeeting.title,
          organizer: currentMeeting.organizer,
          department: currentMeeting.department,
          attendees: currentMeeting.attendees,
          startDateTime: currentMeeting.parsedStartDateTime,
          endDateTime: currentMeeting.parsedEndDateTime
        } : null,
        upcomingMeetings: upcomingMeetings.map(meeting => ({
          title: meeting.title,
          organizer: meeting.organizer,
          department: meeting.department,
          attendees: meeting.attendees,
          startDateTime: meeting.parsedStartDateTime,
          endDateTime: meeting.parsedEndDateTime
        }))
      };
    } catch (error) {
      console.error('Error fetching meetings:', error);
      throw error;
    }
  }
}

// Initialize service with credentials
const sheetsService = new GoogleSheetsService(credentials);

// Initialize the service when server starts
(async () => {
  try {
    await sheetsService.initialize();
    console.log('Server initialization complete');
  } catch (error) {
    console.error('Failed to initialize server:', error);
    process.exit(1);
  }
})();

// API Endpoints
app.get('/api/meetings', async (req, res) => {
  try {
    const meetings = await sheetsService.getMeetings(process.env.SPREADSHEET_ID);
    res.json(meetings);
  } catch (error) {
    console.error('Error fetching meetings:', error);
    res.status(500).json({ 
      error: 'Failed to fetch meetings',
      details: process.env.NODE_ENV === 'development' ? error.message : undefined 
    });
  }
});

// Health check endpoint
app.get('/health', (req, res) => {
  res.json({ status: 'ok', timestamp: new Date().toISOString() });
});

const PORT = process.env.PORT || 3000;
app.listen(PORT, () => {
  console.log(`Server running on port ${PORT}`);
});

// Error handling
process.on('uncaughtException', (error) => {
  console.error('Uncaught Exception:', error);
  process.exit(1);
});

process.on('unhandledRejection', (error) => {
  console.error('Unhandled Rejection:', error);
  process.exit(1);
});
